/// @file deviceManagement.h
///
/// @brief Device manager main header
///
/// This file defines the DeviceInfo class and the enum it uses.
/// This header must not be included directly. Rather include the @ref
/// deviceDetector.h file instead.
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see deviceDetector.h DeviceDetector

#ifndef __DEVICE_MANAGEMENT_H__
#define __DEVICE_MANAGEMENT_H__

#include <string>

namespace adit {
namespace uspi {

/// @enum dev_sysattr
/// @brief System attributes enumeration for Udev.
///
/// The device manager gather specific information from udev.
/// The following enum lists the elements that are gathered, some as
/// plain strings, other as integers.
enum dev_sysattr {
    DSYS_ID_USB_INTERFACES = 0, ///< DSYS_ID_USB_INTERFACES
    DSYS_SERIAL,                ///< The serial number
    DSYS_PRODUCT,               ///< The product name
    DSYS_MANUFACTURER,          ///< The manufacturer name
    DSYS_ADDRESS,               ///< The device address (for %Bluetooth)
    // From here we shall only have integers
    DSYS_IDVENDOR,              ///< The vendor ID
    DSYS_IDPRODUCT,             ///< The product ID
    DSYS_DEVPATH,               ///< The device path size
    DSYS_DEVNUM,                ///< The device number
    DSYS_DEVICE_CLASS,          ///< The device class
    DSYS_DEVICE_SUB_CLASS,      ///< The device subclass
    DSYS_DEVICE_PROTOCOL,       ///< The device protocol
    DSYS_NONE                   ///< End delimiter
};

/// @defgroup DSYS_DEFINITIONS Definitions used to handle dev_sysattr
/// They aim to distinguish the limit between strings dev_sysattr and integers
/// ones.
/// @{
/// @var DSYS_STRINGS
/// The beginning of the strings sys attributes
/// @var DSYS_IDS
/// The beginning of the integers sys attributes
const int DSYS_STRINGS = DSYS_ID_USB_INTERFACES;
const int DSYS_IDS = DSYS_IDVENDOR;
/// @}

/// @brief Device information class
///
/// Stores the device data and provide interface to store,
/// access and check it.
/// The check is done against the device detection class mask.
class DeviceInfo {
    /// The device node if it exists
    std::string mNode;
    /// The device path
    std::string mDevpath;
    /// @var mDevInfo[DSYS_NONE]
    /// Table of strings and integers, to ease comparisons
    /// @struct DevInfoUnion
    /// The DeviceInfo structure containing both string and integer representing
    /// the same value.
    struct DevInfoUnion {
        /// The std::string version of the device info
        std::string msDevInfo;
        /// The integer version of the device info if valid
        uint32_t miDevInfo;

        DevInfoUnion() : msDevInfo(), miDevInfo(0UL) {};
    } mDevInfo[DSYS_NONE];

public:
    DeviceInfo();

    /// The device type mask corresponding to this device
    uint32_t mMask;

    /// Setter for the mDevInfo
    void setDeviceInfo(enum dev_sysattr, const char *);

    /// Getter for the std::string version of mDevInfo
    std::string getDeviceInfo(enum dev_sysattr in) {
        return mDevInfo[in].msDevInfo;
    };

    /// Getter for the integer version of mDevInfo
    uint32_t getiDeviceInfo(enum dev_sysattr in) {
        return mDevInfo[in].miDevInfo;
    };

    /// Setter for the @ref mNode
    void setNode(const char *in) { mNode.assign(in); };
    /// Getter for the @ref mNode
    std::string getNode() { return mNode; };

    /// Setter for the mDevPath
    void setDevPath(const char *in) { mDevpath.assign(in); }
    /// Getter for the mDevPath
    std::string getDevPath() { return mDevpath; };

    DeviceInfo& operator=(const DeviceInfo &di);
    int operator==(const DeviceInfo &di) const;
    int operator!=(const DeviceInfo &di) const;
    int operator<(const DeviceInfo &di) const;

    /// Dumps the device info into the logger
    void dump();

    /// Analyzes the device info and sets the @ref mMask accordingly
    DeviceInfo& check();
};

} // namespace uspi
} // namespace adit
#endif // __DEVICE_MANAGEMENT_H__
